<?php
/*
Plugin Name:  PDEE Scholarships
Description:  Adds an admin interface for scholarships (title, description, short description, application URL) and a [pdee_scholarships] shortcode to show them on the front-end.
Version:      1.0.0
Author:       pdee
License:      GPL-2.0+
*/

if ( ! defined( 'ABSPATH' ) ) {
	exit; // No direct access.
}

class PDEE_Scholarships {

	public function __construct() {
		// Register everything.
		add_action( 'init',                [ $this, 'register_post_type' ] );
		add_action( 'add_meta_boxes',      [ $this, 'add_meta_box' ] );
		add_action( 'save_post',           [ $this, 'save_meta' ], 10, 2 );
		add_shortcode( 'pdee_scholarships',[ $this, 'shortcode' ] );
		add_action( 'wp_enqueue_scripts',  [ $this, 'enqueue_styles' ] );

		// Flush rewrites on (de)activation so “Scholarship” URLs work right away.
		register_activation_hook( __FILE__, [ $this, 'flush_rewrite' ] );
		register_deactivation_hook( __FILE__, [ $this, 'flush_rewrite' ] );
	}

	/*--------------------------------------------------------------
	 *  Post Type & Meta
	 *-------------------------------------------------------------*/
	public function register_post_type() {
		register_post_type(
			'pdee_scholarship',
			[
				'labels' => [
					'name'               => 'Scholarships',
					'singular_name'      => 'Scholarship',
					'add_new'            => 'Add New',
					'add_new_item'       => 'Add New Scholarship',
					'edit_item'          => 'Edit Scholarship',
					'view_item'          => 'View Scholarship',
					'search_items'       => 'Search Scholarships',
					'not_found'          => 'No scholarships found',
					'not_found_in_trash' => 'No scholarships in the trash',
				],
				'public'       => true,          // Gives us the editor & show-in-front-end
				'show_in_menu' => true,          // Adds left-hand admin menu item
				'menu_icon'    => 'dashicons-awards',
				'supports'     => [ 'title', 'editor' ], // Title = scholarship name, Editor = full description
				'show_in_rest' => true,          // Gutenberg & REST API
				'has_archive'  => false,         // No archive page needed
			]
		);
	}

	public function add_meta_box() {
		add_meta_box(
			'pdee_sch_meta',
			'Scholarship Details',
			[ $this, 'meta_box_html' ],
			'pdee_scholarship',
			'normal',
			'default'
		);
	}

	public function meta_box_html( $post ) {
		wp_nonce_field( 'pdee_scholarship_save', 'pdee_scholarship_nonce' );

		$short = get_post_meta( $post->ID, '_pdee_short_desc', true );
		$url   = get_post_meta( $post->ID, '_pdee_apply_url', true );
		?>
		<p>
			<label><strong>Short Description</strong></label><br>
			<textarea name="pdee_short_desc" rows="3" style="width:100%"><?php echo esc_textarea( $short ); ?></textarea>
		</p>
		<p>
			<label><strong>Application URL</strong></label><br>
			<input type="url" name="pdee_apply_url" value="<?php echo esc_attr( $url ); ?>" style="width:100%" placeholder="https://example.com/apply">
		</p>
		<?php
	}

	public function save_meta( $post_id, $post ) {
		// Verify nonce & permissions.
		if ( ! isset( $_POST['pdee_scholarship_nonce'] ) ||
		     ! wp_verify_nonce( $_POST['pdee_scholarship_nonce'], 'pdee_scholarship_save' ) ||
		     defined( 'DOING_AUTOSAVE' ) ||
		     'pdee_scholarship' !== $post->post_type ||
		     ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		update_post_meta( $post_id, '_pdee_short_desc', sanitize_textarea_field( $_POST['pdee_short_desc'] ?? '' ) );
		update_post_meta( $post_id, '_pdee_apply_url',  esc_url_raw( $_POST['pdee_apply_url']  ?? '' ) );
	}

	/*--------------------------------------------------------------
	 *  Front-end
	 *-------------------------------------------------------------*/
	public function enqueue_styles() {
		$css = "
			.pdee-scholarship-item { margin:0 0 2.5rem; padding:0 0 2.5rem; border-bottom:1px solid #eee; }
			.pdee-title           { font-weight:700; font-size:1.4rem; margin:0 0 .5rem; }
			.pdee-date            { color:#777; font-size:.9rem; margin:0 0 .75rem; }
			.pdee-apply           { background:#0073aa; color:#fff; padding:.5rem 1rem; border-radius:3px; text-decoration:none; }
			.pdee-apply:hover     { background:#005177; }
		";
		wp_register_style( 'pdee-scholarships', false );
		wp_enqueue_style(  'pdee-scholarships' );
		wp_add_inline_style( 'pdee-scholarships', $css );
	}

	public function shortcode( $atts ) {
		$atts = shortcode_atts(
			[
				'posts_per_page' => -1,  // show all
				'orderby'        => 'date',
				'order'          => 'DESC',
			],
			$atts,
			'pdee_scholarships'
		);

		$q = new WP_Query( [
			'post_type'      => 'pdee_scholarship',
			'post_status'    => 'publish',
			'posts_per_page' => intval( $atts['posts_per_page'] ),
			'orderby'        => $atts['orderby'],
			'order'          => $atts['order'],
		] );

		if ( ! $q->have_posts() ) {
			return '<p>No scholarships available right now.</p>';
		}

		ob_start();
		echo '<div class="pdee-scholarships">';
		while ( $q->have_posts() ) :
			$q->the_post();
			$url   = get_post_meta( get_the_ID(), '_pdee_apply_url', true );
			?>
				<div class="pdee-scholarship-item">
					<h2 class="pdee-title"><?php the_title(); ?></h2>

					<div class="pdee-description">
						<?php the_content(); // full description ?>
					</div>

					<p class="pdee-date"><?php echo esc_html( get_the_date() ); ?></p>

					<?php if ( $url ) : ?>
						<p><a class="pdee-apply" href="<?php echo esc_url( $url ); ?>" target="_blank" rel="noopener">Apply&nbsp;Here</a></p>
					<?php endif; ?>
				</div>
			<?php
		endwhile;
		wp_reset_postdata();
		echo '</div>';

		return ob_get_clean();
	}

	/*--------------------------------------------------------------*/
	public function flush_rewrite() {
		flush_rewrite_rules();
	}
}

new PDEE_Scholarships;
